/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpP11Key.c,v 1.59.2.12 2001/05/15 21:35:08 ajivsov Exp $
____________________________________________________________________________*/
#include "pgpSDKBuildFlags.h"

/* PKCS-11 (Cryptoki) Interface */

#include "pgpConfig.h"
#include <string.h>
#include <stddef.h>

#include "pgpDebug.h"
#include "pgpKeyMisc.h"
#include "pgpP11Key.h"
#include "pgpTokenLib.h"
#include "bn.h"
#include "pgpCFBPriv.h"
#include "pgpSymmetricCipherPriv.h"
#include "pgpHashPriv.h"
#include "pgpMem.h"
#include "pgpErrors.h"
#include "bnprime.h"
#include "pgpPubKey.h"
#include "pgpRandomX9_17.h"
#include "pgpStr2Key.h"
#include "pgpContext.h"
#include "pgpEnv.h"
#include "pgpKeyPriv.h"
#include "pgpX509Priv.h"
#include "pgpFIPSPriv.h"

#ifdef PGP_WIN32
#include "pgpPktByte.h"
#include "pgpEndianConversion.h"
#endif

/* Size of buffer to hold passphrase */
#define PASSBUFFERSIZE		80
/* Magic number at start of such buffers */
#define PASSBUFMAGIC	0x44201312
/* Mask to hide passphrase in cached buffers */
#define PASSBUFMASK		0x9D

/* A PGPSecKey's priv points to this */
/* This struct is always allocated using PGPNewSecureData */
typedef struct P11Sec
{
	PGPContextRef	context;
	
	PGPToken		*tok;
	int				locked;
	BigNum			mod;		/* RSA Modulus */
	PGPByte			*cryptkey;
	PGPSize			cklen;
	DEBUG_STRUCT_CONSTRUCTOR( P11secPlus )
} P11Sec ;


PGPByte sP11Module[255] = { '\0' };


#if PGP_WIN32
static PGPBoolean sHadTcl = FALSE;

/* PGPtcl library entry points */
static struct PGPtclEntries {
	PGPError (*pgpTokenInit) (void);
	PGPToken *(*pgpGetNthTokenObject) (PGPUInt32 n);
	PGPToken *(*pgpGetTokenObjectByPrivKeyID) (const PGPByte *keyID);
	PGPUInt32 (*pgpCountTokenObjects) (void);
	PGPError (*pgpAcquireAllTokens) (void);
	PGPError (*pgpReleaseAllTokens) (void);
    PGPError (*pgpSetP11DllName)(const PGPByte *name);
    PGPError ( *pgpFreeMem ) (void *p);
	PGPInt32 (*pgpGetTokenNum)(PGPToken *token);
};

static struct PGPtclEntries F;
static inited = FALSE;
static HINSTANCE hTCL = NULL;
static sLoadTclTries = 0;
static int sCustomTcl = -1;

/* Load our PGPtcl(P11) library dynamically */
static void
sLoadTCL()
{
	if( !inited && IsNull(hTCL) )
	{
        PGPBoolean is_generic = FALSE;

        if( sCustomTcl == -1 )  {   /* Try once to load custom TCL */
		    hTCL = LoadLibrary("PGPtcl");
            if( IsNull(hTCL) )   {
                sCustomTcl = 0;
                inited = FALSE;
            }
            else  {
#if PGP_DEBUG
				OutputDebugString("sLoadTCL: custom TCL loaded (PGPtcl.dll)\n");
#endif
                sCustomTcl = 1;
                inited = TRUE;
            }
        }

        if( ! inited ) {
            if( sP11Module[0] != '\0' )  {
				inited = TRUE;  /* Tried -- no matter what the result */
                hTCL = LoadLibrary("PGPtclP11");
                if( IsntNull(hTCL) )  
                    is_generic = TRUE;
            }

        }

		if( IsntNull( hTCL ) )
		{
            memset( &F, 0, sizeof(F) );

			F.pgpTokenInit = (void *)GetProcAddress( hTCL,
					"pgpTokenInit" );
			F.pgpGetNthTokenObject = (void *)GetProcAddress( hTCL,
					"pgpGetNthTokenObject" );
			F.pgpGetTokenObjectByPrivKeyID = (void *)GetProcAddress( hTCL,
					"pgpGetTokenObjectByPrivKeyID" );
			F.pgpCountTokenObjects = (void *)GetProcAddress( hTCL,
					"pgpCountTokenObjects" );
			F.pgpAcquireAllTokens = (void *)GetProcAddress( hTCL,
					"pgpAcquireAllTokens" );
			F.pgpReleaseAllTokens = (void *)GetProcAddress( hTCL,
					"pgpReleaseAllTokens" );
			F.pgpFreeMem = (void *)GetProcAddress( hTCL,
					"pgpFreeMem" );
			F.pgpGetTokenNum = (void *)GetProcAddress( hTCL,
					"pgpGetTokenNum" );
            if( is_generic )
			    F.pgpSetP11DllName = (void *)GetProcAddress( hTCL,
					"pgpSetP11DllName" );
			if( IsNull( F.pgpTokenInit )
				|| IsNull( F.pgpGetNthTokenObject )
				|| IsNull( F.pgpGetTokenObjectByPrivKeyID )
				|| IsNull( F.pgpCountTokenObjects )
				|| IsNull( F.pgpAcquireAllTokens ) 
                || IsNull( F.pgpReleaseAllTokens ) 
                || IsNull( F.pgpFreeMem ) 
				|| IsNull( F.pgpGetTokenNum )
                || ( is_generic && IsNull( F.pgpSetP11DllName )) )
			{
				/* Incompatible library version, pretend it didn't load */
				FreeLibrary(hTCL);
				hTCL = NULL;
				memset( &F, 0, sizeof(F) );
			} else {
                pgpAssert( !is_generic || sP11Module[0] != '\0' );
                if( is_generic && F.pgpSetP11DllName( sP11Module ) != kPGPError_NoErr )  {
#if PGP_DEBUG
					OutputDebugString("PGPtclP11.pgpSetP11DllName() failed\n");
#endif
					FreeLibrary(hTCL);
                    hTCL = NULL;
					memset( &F, 0, sizeof(F) );
				}

                if( hTCL && F.pgpTokenInit() != kPGPError_NoErr )  {
#if PGP_DEBUG
					OutputDebugString("PGPtclP11.pgpTokenInit() failed\n");
#endif
					FreeLibrary(hTCL);
                    hTCL = NULL;
					memset( &F, 0, sizeof(F) );

					if( sLoadTclTries < 5 )  {
						sLoadTclTries ++;
						inited=FALSE;	/* Give more tries */
					}
#if PGP_DEBUG
					else
						OutputDebugString("PGPtclP11.pgpTokenInit() failed. No more tries\n");
#endif

				}
			}
		}
	}
}
#endif /* PGP_WIN32 */

#if PGP_WIN32

/* 
	Create PGP Public Key Packet V3/V4 from the keystub, RSA modulus and 
	public exponent. 

    Input size in bytes.
	Caller must free the pkt_out.
*/
static PGPError 
sPgpPubKeyPktFromRsa( PGPContextRef context, 
        const PGPByte *mod_buf, PGPSize mod_size, 
        const PGPByte *exp_buf, PGPSize exp_size, 
        const pgpTokenPubKeyStub *keyStub, 
	    PGPByte **pkt_out, PGPSize *pkt_len_out )
{
#pragma pack(push, cryptoki, 1)
	struct RsaPktHdr {
		PGPByte		header;					/* 0 offset */
		PGPUInt16	total_len;				/* 1 */
		PGPByte		ver;	     			/* 3 */
		PGPUInt32	time;	         		/* 4 */
		union {
			PGPUInt16	expiration;				/* 8 */
			PGPByte		alg_v4;					/* 8 */
		};
		PGPByte		alg_v3;					/* 10 */
											/* 11 */
	};
	struct RsaPktHdr *rsa_pkt_hdr;
#pragma pack(pop, cryptoki)
	const v = keyStub->flags & 0xf;
	const struct_size = sizeof(struct RsaPktHdr) - ( v==PGPVERSION_3 ? 0 : 2/*no expiration*/ );

	int total_size = mod_size+exp_size + struct_size + 2*sizeof(PGPUInt16)/*MPI length*/; 
	PGPByte *pkt_out_temp;
	PGPByte *p;
	PGPUInt16 size;

	pgpAssert( mod_size < 0x10000 && exp_size < 0x10000 );
	pgpAssert( keyStub && (keyStub->flags & 0x80) == 0 ); /* It can be full key later */

	*pkt_out = NULL;
	*pkt_len_out = 0;

	pkt_out_temp = (PGPByte*)pgpContextMemAlloc( context, total_size, 0 );

	if( pkt_out_temp == NULL )   
		return kPGPError_OutOfMemory;

	pgpAssert( v == PGPVERSION_3 || v == PGPVERSION_4 );

	rsa_pkt_hdr = (struct RsaPktHdr *)pkt_out_temp;

	size = total_size-sizeof(rsa_pkt_hdr->header)-sizeof(rsa_pkt_hdr->total_len);

	rsa_pkt_hdr->header = PKTBYTE_BUILD( PKTBYTE_PUBKEY, 1/*two octet length*/ ); 
	PGPUInt16ToEndian(size, kPGPBigEndian, (PGPByte*)&(rsa_pkt_hdr->total_len)); 
	rsa_pkt_hdr->ver = 	v;
	rsa_pkt_hdr->time = keyStub->creationtime;
	if( v == PGPVERSION_3 )  {
		rsa_pkt_hdr->alg_v3 = kPGPPublicKeyAlgorithm_RSA;
		rsa_pkt_hdr->expiration = 0;	/*never*/
	}
	else
		rsa_pkt_hdr->alg_v4 = kPGPPublicKeyAlgorithm_RSA;

	p = pkt_out_temp + struct_size;

	size = mod_size * 8;	
	pgpAssert( *(PGPByte *)mod_buf & 0x80 ); 
	PGPUInt16ToEndian(size, kPGPBigEndian, p);	/* in bits */
	p += sizeof(PGPUInt16);

	pgpCopyMemory( mod_buf, p, mod_size );
	p += mod_size;

	size = exp_size * 8;
	{	/* Determine the actual bit size */
		int byte = *(PGPByte *)exp_buf; /* Highest byte */

		if( byte == 0 )  {
			free( pkt_out_temp );
			return kPGPError_CorruptData;
		}

		while( (byte & 0x80) == 0 )  {
			size --;
			byte <<= 1;
		}
	}
	PGPUInt16ToEndian(/*in*/size, kPGPBigEndian, /*out*/p);	/* in bits */
	p += sizeof(PGPUInt16);

	pgpCopyMemory( exp_buf, p, exp_size );

	*pkt_out = (PGPByte *)pkt_out_temp;
	*pkt_len_out = total_size;

	return kPGPError_NoErr;
}

#endif /* WIN32 */


#if PGP_WIN32

static PGPByte *
sPgpPubKeyPktFromDataInfo( PGPContextRef context, const pgpTokenDataInfo *di, PGPSize *size )  {
    const PGPByte *e = (PGPByte*)(di+1);
    const PGPByte *m = e + di->exp_size;
    PGPByte *out;

    sPgpPubKeyPktFromRsa( context, m, di->mod_size, e, di->exp_size, 
                          &(di->pubKeyStub), &out, size );

    return out;
}

#endif /* WIN32 */

/** Public key functions **/

/*
 * Not implemented
 */
PGPPubKey *
p11PubFromBuf(
	PGPContextRef	context,
	PGPByte const *	buf,
	PGPSize			size,
	PGPError *		error)
{
	(void) context;
	(void) buf;
	(void) size;
	(void) error;
	*error = kPGPError_PublicKeyUnimplemented;
	return NULL;
}



/** Secret key functions **/

static void
p11SecDestroy(PGPSecKey *seckey)
{
	P11Sec *sec = (P11Sec *)seckey->priv;
	PGPContextRef	context;

	pgpAssertAddrValid( seckey, PGPSecKey );
	context	= seckey->context;

	/* Make sure token is locked when we free it */
	(void)pgpTokenObjDeAuth( sec->tok );

	PGPFreeData( sec );			/* Wipes as it frees */
	pgpClearMemory( seckey,  sizeof(seckey));
	pgpContextMemFree( context, seckey);
}

/*
 * Generate a PGPPubKey from a PGPSecKey
 */
static PGPPubKey *
p11Pubkey(PGPSecKey const *seckey)
{
	P11Sec *sec = (P11Sec *)seckey->priv;
	PGPError error;

	return pgpPubKeyFromBuf( seckey->context, seckey->pkAlg,
							 sec->cryptkey, sec->cklen, &error );
}

/*
 * Set keyid
 */
static void
p11SecSetKeyID(PGPSecKey *seckey, PGPByte *keyid)
{
#if PGP_WIN32
	P11Sec *sec = (P11Sec *)seckey->priv;
	/* Change keyid on the token */
	pgpTokenObjEditKeyList( sec->tok, seckey->keyID, keyid );
#endif
	pgpCopyMemory(keyid, seckey->keyID, sizeof(seckey->keyID));
}

/*
 * Yes, there *is* a reason that this is a function and no a variable.
 * On a hardware device with an automatic timeout,
 * it actually might need to do some work to find out.
 */
static int
p11Islocked(PGPSecKey const *seckey)
{
	P11Sec const *sec = (P11Sec *)seckey->priv;

	return sec->locked;
}


/*
 * Return the algorithm and (symmetric) key size used for locking/unlocking
 * the secret key.
 * This is not possible with token keys, they are locked in hardware.
 * We return a buffer size so that we can fake the passphrase conversion
 */
static PGPError
p11LockingAlgorithm(
	PGPSecKey const *seckey,
	PGPCipherAlgorithm *pAlg,
	PGPSize *pAlgKeySize
	)
{
	(void) seckey;
	if( IsntNull( pAlg ) )
		*pAlg = kPGPCipherAlgorithm_None;
	if( IsntNull( pAlgKeySize ) )
		*pAlgKeySize = PASSBUFFERSIZE;
	return kPGPError_NoErr;
}


/*
 * Return the StringToKey type for unlocking the given key.  We use
 * kPGPStringToKey_Literal to flag a secret split unlocking buffer.
 * Returns kPGPStringToKey_Simple if key has no passphrase.
 */
static PGPError
p11S2KType(
	PGPSecKey const *seckey,
	PGPStringToKeyType *s2kType
	)
{
	(void) seckey;
	(void) s2kType;
	return kPGPError_PublicKeyUnimplemented;
}


/* Recover passphrase from hashed version */
static PGPError
sRecoverHashedPhrase( PGPByte const *hphrase, PGPSize hphraseLen,
	char *pphrase, PGPSize *pplen )
{
	PGPByte mask = 0;
	PGPSize plen;

	if( hphraseLen < sizeof(PGPUInt32)+1
		|| *(PGPUInt32 *)hphrase != PASSBUFMAGIC )
		return kPGPError_BadPassphrase;
	hphrase += sizeof( PGPUInt32 );
	*pplen = plen = (PGPSize)*hphrase++;
	while( plen-- )
		*pphrase++ = *hphrase++ ^ (mask+=PASSBUFMASK);
	return kPGPError_NoErr;
}

/*
 * Convert a passphrase into a s2k literal buffer for the key.
 * Returns error code.  Output buffer will be size of the *pAlgKeySize
 * parameter from pgpSecKeyLockingalgorithm.
 * We store passphrase, preceded by length, lightly masked
 */
static PGPError
p11ConvertPassphrase(PGPSecKey *seckey, PGPEnv const *env,
	  char const *phrase, PGPSize plen, PGPByte *outbuf)
{
	PGPByte mask = 0;

	(void) env;
	(void) seckey;

	plen = pgpMin( plen, PASSBUFFERSIZE - 1 - sizeof(PGPUInt32) );

	*(PGPUInt32 *)outbuf = PASSBUFMAGIC;
	outbuf += sizeof(PGPUInt32);
	*outbuf++ = (PGPByte) plen;
	while( plen-- )
		*outbuf++ = (PGPByte)*phrase++ ^ (mask+=PASSBUFMASK);
	return kPGPError_NoErr;
}


/*
 * Try to unlock the secret key wih the given passphrase.  Returns >0
 * if it was the correct passphrase. =0 if it was not, and <0 on error.
 * Does not alter the key even if it's the wrong passphrase and already
 * unlocked.  A NULL passphrae will work if the key is unencrypted.
 */

static int
p11Unlock(PGPSecKey *seckey,
	  char const *phrase, PGPSize plen, PGPBoolean hashedPhrase)
{
	P11Sec *sec = (P11Sec *)seckey->priv;
	char passbuf[PASSBUFFERSIZE];
	int rslt;

	if( pgpFIPSModeEnabled() )
	{
		/* FIPS does not allow using keys with NULL passphrases */
		if( IsNull( phrase ) || plen == 0 )
			return( 0 );
	}
	
	if( hashedPhrase )
	{
		if( IsPGPError( sRecoverHashedPhrase( (const unsigned char *) phrase, plen, passbuf, &plen )))
			return 0;
		phrase = passbuf;
	}

	rslt = pgpTokenObjAuth( sec->tok, phrase, plen );
	if( rslt != 0 )
		return 0;		/* Failure */

	sec->locked = 0;
	return 1;
}

/*
 * Relock the key.
 */
static void
p11Lock(PGPSecKey *seckey)
{
	P11Sec *sec = (P11Sec *)seckey->priv;

	(void)pgpTokenObjDeAuth( sec->tok );
	sec->locked = 1;
}

static PGPSize
p11SecMaxdecrypted(PGPSecKey const *seckey, PGPPublicKeyMessageFormat format);


/*
 * Try to decrypt the given esk.  If the key is locked, try the given
 * passphrase.  It may or may not leave the key unlocked in such a case.
 * (Some hardware implementations may insist on a password per usage.)
 */
static int
p11Decrypt(PGPSecKey *seckey, PGPByte const *esk, PGPSize esklen,
		   PGPByte *key, PGPSize *keylen,
		   char const *phrase, PGPSize plen,
		   PGPPublicKeyMessageFormat format)
{
#if PGP_DECRYPT_DISABLE /* [ */

	(void)seckey;
	(void)esk;
	(void)esklen;
	(void)key;
	(void)keylen;
	(void)phrase;
	(void)plen;
	(void)format;
	return kPGPError_FeatureNotAvailable;

#else /* PGP_DECRYPT_DISABLE */  /* ]  [ */

	P11Sec *sec = (P11Sec *)seckey->priv;
	BigNum bn;
	PGPInt32 i;
	PGPInt32 j;
	PGPUInt32 t;
	PGPUInt32 modbytes;
	PGPMemoryMgrRef		mgr	= NULL;

	mgr	= PGPPeekContextMemoryMgr( seckey->context );

	if (sec->locked) {
		i = p11Unlock(seckey, phrase, plen, FALSE);
		if (i <= 0)
			return i ? i : kPGPError_KeyIsLocked;
		pgpAssert(!sec->locked);
	}

	if (esklen < 2)
		return kPGPError_BadSessionKeySize;
	
	bnBegin(&bn, mgr, TRUE);
	modbytes = bnBytes( &sec->mod );
	i = pgpBnGetFormatted(&bn, esk, esklen, modbytes, format);
	if (i <= 0)
		return kPGPError_BadSessionKeySize;

	bnExtractBigBytes( &bn, key, 0, modbytes );
	bnEnd(&bn);

	i = pgpTokenObjDecrypt( sec->tok, (void *)seckey->keyID, key, modbytes,
							key, modbytes );
	if (i < 0)
		return i;

	if ((PGPSize)i > modbytes || i < 3)
		return kPGPError_CorruptData;

	if (format == kPGPPublicKeyMessageFormat_PGP) {
		/* Check checksum (should this be here?) */
		t = 0;
		for (j = 1; j < i-2; j++)
			t += key[j];
		if (t != ((unsigned)key[i-2]<<8) + key[i-1])
			return kPGPError_CorruptData;
		pgpClearMemory(key+i-2, 2);

		/* The actual key */
		if (keylen)
			*keylen = (PGPSize)i-2;
	} else {
		/* The actual key */
		if (keylen)
			*keylen = (PGPSize)i;
	}

	return 0;

#endif /* PGP_DECRYPT_DISABLE */ /* ] */
}


/*
 * Return the size of the buffer needed, worst-case, for the decrypted
 * output.  This will be the size of the "key" parameter to p11Decrypt.
 * We use that as a temp buffer to hold the raw decrypted value so it must
 * be the same size as the modulus.
 */
static PGPSize
p11SecMaxdecrypted(PGPSecKey const *seckey, PGPPublicKeyMessageFormat format)
{
	P11Sec const *sec = (P11Sec *)seckey->priv;

	(void) format;

	return bnBytes(&sec->mod);
}

/* Return the largest possible PGPESK size for a given key */
static PGPSize
p11SecMaxesk(PGPSecKey const *seckey, PGPPublicKeyMessageFormat format)
{
	P11Sec const *sec = (P11Sec *)seckey->priv;

	if (format == kPGPPublicKeyMessageFormat_PGP)
		return 2 + bnBytes(&sec->mod);
	else if (format == kPGPPublicKeyMessageFormat_PKCS1 ||
			 format == kPGPPublicKeyMessageFormat_X509  ||
			 format == kPGPPublicKeyMessageFormat_IKE)
		return bnBytes(&sec->mod);

	pgpAssert(0);
	return 0;
}

static PGPSize
p11SecMaxsig(PGPSecKey const *seckey, PGPPublicKeyMessageFormat format)
{
	P11Sec const *sec = (P11Sec *)seckey->priv;

	if (format == kPGPPublicKeyMessageFormat_PGP)
		return 2 + bnBytes(&sec->mod);
	else if (format == kPGPPublicKeyMessageFormat_PKCS1 ||
			 format == kPGPPublicKeyMessageFormat_IKE ||
			 format == kPGPPublicKeyMessageFormat_X509)
		return bnBytes(&sec->mod);

	pgpAssert(0);
	return 0;
}

static int
p11Sign(PGPSecKey *seckey, PGPHashVTBL const *h, PGPByte const *hash,
	PGPByte *sig, PGPSize *siglen, PGPRandomContext const *rc,
	PGPPublicKeyMessageFormat format)
{
#if PGP_SIGN_DISABLE /* [ */

	(void)seckey;
	(void)h;
	(void)hash;
	(void)sig;
	(void)siglen;
	(void)rc;
	(void)format;
	return kPGPError_FeatureNotAvailable;

#else

	P11Sec *sec = (P11Sec *)seckey->priv;
	BigNum bn;
	int i;
	unsigned t;
	PGPUInt32 modbytes;
	PGPMemoryMgrRef		mgr	= NULL;
	
	mgr	= PGPPeekContextMemoryMgr( seckey->context );

	/* We don't need these arguments, although other algorithms may... */
	(void)rc;
	(void)format;

	if (sec->locked)
		return kPGPError_KeyIsLocked;

	bnBegin(&bn, mgr, TRUE);

	modbytes = bnBytes( &sec->mod );

	t = h->DERprefixsize;
	/* IKE does not put in hash OID */
	if (format == kPGPPublicKeyMessageFormat_IKE)
		t = 0;
	if (t+h->hashsize  > modbytes)
		return kPGPError_PublicKeyTooSmall;
	pgpClearMemory( sig,  t);
	memcpy(sig, h->DERprefix, t);
	memcpy(sig+t, hash, h->hashsize);
	t += h->hashsize;

	i = pgpTokenObjSign( sec->tok, seckey->keyID, sig, t, sig, modbytes );
	if (bnInsertBigBytes(&bn, sig, 0, modbytes) < 0)
		return kPGPError_OutOfMemory;
	pgpClearMemory( sig,  t);

	if (i >= 0) {
		t = 0;
		if (format == kPGPPublicKeyMessageFormat_X509) {
			/* Output unformatted, but with no leading zeros */
			format = kPGPPublicKeyMessageFormat_PKCS1;
			t += pgpBnPutFormatted(&bn, sig+t, bnBytes(&bn), format);
		} else {
			t += pgpBnPutFormatted(&bn, sig+t, bnBytes(&sec->mod), format);
		}
		if (siglen)
			*siglen = (PGPSize)t;
		i = 0;
	}
	bnEnd(&bn);
	return i;

#endif /* PGP_SIGN_DISABLE */ /* ] */
}


/*
 * Change passphrase on token.  Not implemented.
 */

static int
p11ChangeLock(PGPSecKey *seckey, PGPEnv const *env, 
	PGPRandomContext const *rc, char const *ophrase, PGPSize oplen,
	PGPBoolean oHashedPhrase, char const *phrase, PGPSize plen,
	PGPStringToKeyType s2ktype)
{
	P11Sec *sec = (P11Sec *)seckey->priv;
	char passbuf[PASSBUFFERSIZE];
	PGPError err;

	(void) env;
	(void) rc;
	(void) s2ktype;

	if( oHashedPhrase )
	{
		if( IsPGPError( err = sRecoverHashedPhrase( (const unsigned char *) ophrase, oplen,
													passbuf, &oplen ) ) )
			return err;
		ophrase = passbuf;
	}

	err = pgpTokenObjSetPIN( sec->tok, (char *)ophrase, oplen,
							 (char *)phrase, plen );
	
	pgpPurgePassphraseCache_back(seckey->context);

	return err;
}

static PGPSize
p11SecBufferLength(PGPSecKey const *seckey)
{
	(void) seckey;
	return 0;
}

static void
p11SecToBuffer(PGPSecKey const *seckey, PGPByte *buf)
{
	(void) seckey;
	(void) buf;
}


static int
p11SecParams(PGPSecKey const *seckey, BigNum *x1, BigNum *x2,
	BigNum *x3, BigNum *x4, BigNum *x5, BigNum *x6, BigNum *x7, BigNum *x8)
{
	(void) seckey;
	(void)x1;
	(void)x2;
	(void)x3;
	(void)x4;
	(void)x5;
	(void)x6;
	(void)x7;
	(void)x8;
	return kPGPError_PublicKeyUnimplemented;
}


/* Fill in secret key structure */
static void
p11FillSecKey(PGPSecKey *seckey, P11Sec *sec)
{
	seckey->priv	            = sec;
	seckey->destroy             = p11SecDestroy;
	seckey->pubkey              = p11Pubkey;
	seckey->islocked            = p11Islocked;
	seckey->lockingalgorithm    = p11LockingAlgorithm;
	seckey->s2ktype             = p11S2KType;
	seckey->convertpassphrase   = p11ConvertPassphrase;
	seckey->unlock              = p11Unlock;
	seckey->lock                = p11Lock;
	seckey->decrypt             = p11Decrypt;
	seckey->maxdecrypted        = p11SecMaxdecrypted;
	seckey->maxsig              = p11SecMaxsig;
	seckey->maxesk              = p11SecMaxesk;
	seckey->sign                = p11Sign;
	seckey->changeLock          = p11ChangeLock;
	seckey->bufferLength        = p11SecBufferLength;
	seckey->toBuffer            = p11SecToBuffer;
	seckey->secparams			= p11SecParams;
	seckey->setkeyid			= p11SecSetKeyID;
}


/* Create a new PGPSecKey object.

 */
PGPSecKey *
p11SecFromBuf(
	PGPContextRef	context,
	PGPByte const *	buf,
	PGPSize			size,
	PGPBoolean		v3,
	void *			tok,
	PGPError *		error)
{
	PGPSecKey *seckey;
	P11Sec *sec;
	PGPError	err	= kPGPError_OutOfMemory;
	PGPMemoryMgrRef		mgr	= PGPPeekContextMemoryMgr( context );
	PGPByte *cryptk;

	(void) v3;

	bnInit();

	/* Holds public part */
	cryptk = (PGPByte *)pgpContextMemAlloc(context,
		size, kPGPMemoryMgrFlags_Clear);
	if (cryptk) {
		sec = (P11Sec *)PGPNewSecureData( mgr, sizeof(*sec), 0 );
		if (sec) {
			pgpClearMemory( sec, sizeof(*sec) );
			sec->context	= context;
			seckey = (PGPSecKey *) pgpContextMemAlloc(context,
						sizeof(*seckey), kPGPMemoryMgrFlags_Clear);
			if (seckey) {
				PGPUInt32 v;

				seckey->context	= context;
				seckey->token	= TRUE;
				pgpCopyMemory( buf, cryptk, size );
				sec->cryptkey = cryptk;
				sec->cklen = size;

				bnBegin(&sec->mod, mgr, FALSE);
				sec->locked = 1;
				seckey->priv = sec;

				/* Hold pointer to token object */
				sec->tok = tok;

				/* Parse buf to get modulus */
				if (size < 5) {
					*error = kPGPError_KeyPacketTruncated;
					return NULL;
				}
				v = ((unsigned)buf[0] << 8) + buf[1];
				v = (v+7)/8;
				if (size < 5+v) {
					*error = kPGPError_KeyPacketTruncated;
					return NULL;
				}
				if (bnInsertBigBytes(&sec->mod, buf+2, 0, v) < 0) {
					*error = kPGPError_OutOfMemory;
				}

				p11FillSecKey(seckey, sec);
				*error = kPGPError_NoErr;
				return seckey;	/* Success! */
			}
			/* Error, clean up and exit */
			PGPFreeData( sec );			/* Wipes as it frees */
		}
	}
	*error = err;
	return NULL;
}


PGPSecKey *
p11SecGenerate(
	PGPContextRef context, PGPByte pkalg,
	unsigned bits, PGPBoolean fastgen, PGPBoolean v3,
	int progress(void *arg, int c), void *arg,
	void *tokptr, PGPByte const *passphrase, PGPSize passphraseLength,
	PGPBoolean genMaster, PGPError *error)
{
#if PGP_WIN32
	PGPSecKey *seckey;
	PGPMemoryMgrRef mgr;
	PGPToken *tok = tokptr;
	PGPByte tmpkeyid[8];
	PGPUInt32 modlen;
	PGPUInt32 pubbuflen;
	PGPUInt32 off;
	PGPByte *modbuf = NULL;
	PGPByte *pubbuf = NULL;
	PGPUInt32 pubexp;
	PGPError err = kPGPError_NoErr;
	BigNum mod, exp;

	pgpAssert( IsntNull( tok ) );

	/* Unlock token */
	err = pgpTokenObjAuth( tok, passphrase, passphraseLength );
	if( err != 0 )
	{
		*error = kPGPError_BadPassphrase;
		return NULL;
	}

	pgpFillMemory( tmpkeyid, 8, 0x55 );	/* Temporary keyid value */

	mgr	= PGPPeekContextMemoryMgr( context );
	bnBegin( &mod, mgr, FALSE );
	bnBegin( &exp, mgr, FALSE );

	modlen = (bits+7)/8;
	modbuf = PGPNewData( mgr, modlen, 0 );

	if (progress && IsPGPError( err = progress(arg, '.') ) )
		goto error;

	err = pgpTokenObjKeygen(tok, tmpkeyid, bits, genMaster, modbuf, &pubexp);
	if( IsPGPError( err ) )
		goto error;

	if (progress && IsPGPError( err = progress(arg, '*') ) )
		goto error;

	if (progress && IsPGPError( err = progress(arg, ' ') ) )
		goto error;

	if (progress && IsPGPError( err = progress(arg, '*') ) )
		goto error;

	bnInsertBigBytes( &mod, modbuf, 0, modlen );
	bnSetQ( &exp, pubexp );
	PGPFreeData( modbuf ); modbuf = NULL;

	pubbuflen = 2 + bnBytes(&mod) + 2 + bnBytes(&exp);
	pubbuf = PGPNewData( mgr, pubbuflen, 0 );

	off = pgpBnPutFormatted(&mod, pubbuf, bnBytes(&mod),
						  kPGPPublicKeyMessageFormat_PGP);
	pgpBnPutFormatted(&exp, pubbuf+off, bnBytes(&exp),
						  kPGPPublicKeyMessageFormat_PGP);
	bnEnd( &mod );
	bnEnd( &exp );
	
	seckey = p11SecFromBuf( context, pubbuf, pubbuflen, FALSE, tok, error );
	PGPFreeData( pubbuf ); pubbuf = NULL;
	pgpAssert( IsntNull( seckey ) );

	pgpCopyMemory( tmpkeyid, seckey->keyID, sizeof(seckey->keyID) );
	
	return seckey;

 error:

	if( IsntNull( modbuf ) )
		PGPFreeData( modbuf );
	if( IsntNull( pubbuf ) )
		PGPFreeData( pubbuf );
	bnEnd( &mod );
	bnEnd( &exp );

	*error = err;
	return NULL;

#else
	(void) context;
	(void) pkalg;
	(void) bits;
	(void) fastgen;
	(void) v3;
	(void) progress;
	(void) arg;
	(void) tokptr;
	(void) passphrase;
	(void) passphraseLength;
	(void) genMaster;
	(void) error;
	return NULL;
#endif
}


static PGPError
sTokenCopyPrivKeyToToken( PGPContextRef context, PGPKeyDBObjRef key,
						PGPBoolean isMaster,
						PGPByte const *passphrase, PGPSize passphraseLength,
						PGPBoolean hashedPhrase, PGPByte const *PIN,
						PGPSize PINlength, PGPUInt32 toknumber )
{
#if PGP_WIN32
	PGPToken *tok;
	PGPByte pkalg;
	PGPKeyID keyid;
	PGPByte *keyidbytes;
	PGPMemoryMgrRef		mgr	= NULL;
	BigNum mod, exp, decexp, p1, p2, coef, decp1, decp2;
	PGPByte *modbuf, *expbuf, *decexpbuf, *p1buf, *p2buf, *coefbuf,
		*decp1buf, *decp2buf;
	PGPSize modlen, explen;
	PGPPubKey *pub;
	PGPSecKey *sec;
	PGPError err = kPGPError_NoErr;

	pgpAssert( OBJISKEY( key ) );

	tok = F.pgpGetNthTokenObject( toknumber );
	mgr	= PGPPeekContextMemoryMgr( context );

	pgpKeyID8( key, &pkalg, &keyid );
	if( pkalg != kPGPPublicKeyAlgorithm_RSA )
		return kPGPError_PublicKeyUnimplemented;

	keyidbytes = (PGPByte *)pgpGetKeyIDBytes( &keyid );

	bnBegin( &mod, mgr, FALSE );
	bnBegin( &exp, mgr, FALSE );
	bnBegin( &decexp, mgr, TRUE );
	bnBegin( &p1, mgr, TRUE );
	bnBegin( &p2, mgr, TRUE );
	bnBegin( &coef, mgr, TRUE );
	bnBegin( &decp1, mgr, TRUE );
	bnBegin( &decp2, mgr, TRUE );

	sec = pgpSecSecKey( key, 0 );
	pub = pgpSecKeyPubkey( sec );
	pgpPubKeyParams( pub, &mod, &exp, NULL, NULL, NULL, NULL, NULL, NULL );
	pgpPubKeyDestroy( pub );
	err = pgpSecKeyUnlock( sec, passphrase, passphraseLength, hashedPhrase );
	if( err <= 0 )
		return kPGPError_BadPassphrase;
	pgpSecKeyParams( sec, &decexp, &p2, &p1, &coef, NULL, NULL, NULL, NULL );
	pgpSecKeyDestroy( sec );

	bnSubQ( &p1, 1 );
	bnMod( &decp1, &decexp, &p1 );
	bnAddQ( &p1, 1 );
	bnSubQ( &p2, 1 );
	bnMod( &decp2, &decexp, &p2 );
	bnAddQ( &p2, 1 );

	modlen = bnBytes( &mod );
	explen = bnBytes( &exp );

	if( modlen % 2 != 0 )
		++modlen;
	
	modbuf = PGPNewData( mgr, modlen, 0 );
	expbuf = PGPNewData( mgr, explen, 0 );
	decexpbuf = PGPNewSecureData( mgr, modlen, 0 );
	p1buf = PGPNewSecureData( mgr, modlen/2, 0 );
	p2buf = PGPNewSecureData( mgr, modlen/2, 0 );
	decp1buf = PGPNewSecureData( mgr, modlen/2, 0 );
	decp2buf = PGPNewSecureData( mgr, modlen/2, 0 );
	coefbuf = PGPNewSecureData( mgr, modlen/2, 0 );

	bnExtractBigBytes( &mod, modbuf, 0, modlen );
	bnExtractBigBytes( &exp, expbuf, 0, explen );
	bnExtractBigBytes( &decexp, decexpbuf, 0, modlen );
	bnExtractBigBytes( &p1, p1buf, 0, modlen/2 );
	bnExtractBigBytes( &p2, p2buf, 0, modlen/2 );
	bnExtractBigBytes( &decp1, decp1buf, 0, modlen/2 );
	bnExtractBigBytes( &decp2, decp2buf, 0, modlen/2 );
	bnExtractBigBytes( &coef, coefbuf, 0, modlen/2 );

	bnEnd( &mod );
	bnEnd( &exp );
	bnEnd( &decexp );
	bnEnd( &p1 );
	bnEnd( &p2 );
	bnEnd( &decp1 );
	bnEnd( &decp2 );
	bnEnd( &coef );

	/* Unlock token */
	err = pgpTokenObjAuth( tok, (IsNull(PIN)?passphrase:PIN),
						   (IsNull(PIN)?passphraseLength:PINlength) );
	if( err != 0 )
		return kPGPError_BadPassphrase;

	err = pgpTokenObjPutPrivKey ( tok, isMaster, keyidbytes, modbuf, modlen,
								  expbuf, explen, decexpbuf, modlen,
								  p1buf, modlen/2, p2buf, modlen/2,
								  decp1buf, modlen/2, decp2buf, modlen/2,
								  coefbuf, modlen/2 );
	
	PGPFreeData( modbuf );
	PGPFreeData( expbuf );
	PGPFreeData( decexpbuf );
	PGPFreeData( p1buf );
	PGPFreeData( p2buf );
	PGPFreeData( decp1buf );
	PGPFreeData( decp2buf );
	PGPFreeData( coefbuf );

	if( IsntPGPError( err ) )
	{
		pgpSyncTokenToKeyDB( context, NULL, TRUE );
	}

	return err;
#else
	(void) context;
	(void) key;
	(void) toknumber;
	(void) isMaster;
	(void) passphrase;
	(void) passphraseLength;
	(void) hashedPhrase;
	(void) PIN;
	(void) PINlength;
	return kPGPError_PublicKeyUnimplemented;
#endif
}

	PGPError
pgpTokenCopyPrivKeyToToken( 
	PGPContextRef context, PGPKeyDBObjRef key,
	PGPBoolean isMaster,
	PGPByte const *passphrase, PGPSize passphraseLength,
	PGPBoolean hashedPhrase, PGPByte const *PIN, PGPSize PINlength,
	PGPUInt32 tokNumber )
{
	PGPKeyDBObjRef keyToUse = key;	

	if( !isMaster && pgpKeyVersion(key) == PGPVERSION_4 )  {
		PGPError err = PGPGetKeyForUsage( 
			key, kPGPKeyPropertyFlags_UsageEncrypt, &keyToUse );

		if( IsPGPError(err) )
			return err;
	}

	return sTokenCopyPrivKeyToToken( context, keyToUse, isMaster, 
		passphrase, passphraseLength, hashedPhrase, PIN, PINlength,tokNumber );
}

#if PGP_WIN32
/*
    Exports all identities with signatures into the certblob, 
	excluding the public key itself. 
	Caller must free the certblob.
 */
static PGPError 
sPgpKeyToMem( PGPContextRef context, PGPMemoryMgrRef mgr,
			    PGPKeyDBObjRef key, 
				PGPByte **certblob, PGPSize *certbloblen,
				PGPKeyDBObjProperty prop )
{
	PGPError err = kPGPError_NoErr;
	PGPKeySetRef oneset;
	PGPFilterRef filter;
	PGPKeySetRef filteredSet;
	PGPFile *pfile;

	*certblob = NULL;
	*certbloblen = 0;

	PGPNewOneKeySet( key, &oneset );
	if( prop != kPGPKeyDBObjProperty_Invalid )  {
		PGPKeyID keyid;
		PGPByte pkalg_notused;

		pgpKeyID8( key, &pkalg_notused, &keyid );

		PGPNewKeyDBObjDataFilter( context, prop,
							  &keyid, sizeof(keyid),
							  kPGPMatchCriterion_Equal, &filter );
		PGPFilterChildObjects( filter, TRUE );
		PGPFilterKeySet( oneset, filter, &filteredSet );
		PGPFreeKeySet( oneset );
	}
	else  {
		filteredSet = oneset;
	}
	pfile = pgpFileMemOpen( context, NULL, 0 );
	pgpExportToPGPFile( filteredSet, pfile, 
		                FALSE, FALSE, FALSE, FALSE /*includepubkey*/ );
	PGPFreeKeySet( filteredSet );
	*certbloblen = (PGPSize)pgpFileTell (pfile );
	*certblob = (PGPByte *)PGPNewData( mgr, *certbloblen, 0 );
	pgpFileSeek( pfile, 0, SEEK_SET );
	pgpFileRead( *certblob, *certbloblen, pfile );
	pgpFileClose( pfile );

	return err;
}

	static PGPError
sPutPubData(
	PGPToken		*tok, 
	PGPByte			*keyid,
	PGPByte			*data,
	PGPSize			size, 
	PGPUInt32		keyCrTime_netorder,
	PGPByte			v )  
{
	pgpTokenDataInfo di = { { 0 } };

	di.pubKeyStub.creationtime = keyCrTime_netorder;

	pgpAssert(v >= 0 && v <= PGPVERSION_4); 
	if( v < PGPVERSION_3 )
	    v = PGPVERSION_3;
	di.pubKeyStub.flags = v & 0xf;

	return pgpTokenObjPutPubKeyData ( tok, keyid, data, size, &di );
}
#endif


	PGPError
pgpTokenCopyPubKeyToToken(
	PGPContextRef		context,
	PGPKeyDBObjRef		key,
	PGPUInt32			toknumber, 
	PGPToken			*tok )
{
#if PGP_WIN32
	PGPByte				pkalg;
	PGPKeyID			keyid;
	PGPByte				*keyidbytes;
	PGPMemoryMgrRef		mgr	= NULL;
	PGPByte				*certblob;
	PGPSize				certbloblen;
	PGPUInt32			keyCrTime_netorder;
	PGPError			err = kPGPError_NoErr;
	PGPByte				v;

	pgpAssert( OBJISKEY( key ) );
	pgpAssert( (toknumber == -1 && tok) ||
			   (toknumber != -1 && tok && 
				     tok == F.pgpGetNthTokenObject(toknumber)) ||
			   (toknumber != -1 && !tok ) );

	if( !tok )
		tok = F.pgpGetNthTokenObject( toknumber );
	mgr	= PGPPeekContextMemoryMgr( context );

	pgpKeyID8( key, &pkalg, &keyid );
	if( pkalg != kPGPPublicKeyAlgorithm_RSA )
		return kPGPError_PublicKeyUnimplemented;

	keyidbytes = (PGPByte *)pgpGetKeyIDBytes( &keyid );
	v = pgpKeyVersion( key );

	/* Extract certificate without the Public key - only child Objects. 
	   Public key will be reconstructed from other objects on the token.
	   We only need key creation time (since it is hashed in PGP certs) 
	   and packet version.
	*/
	PGPUInt32ToEndian(pgpKeyCreation( key ), kPGPBigEndian, (PGPByte*)&keyCrTime_netorder);
	err = sPgpKeyToMem( context, mgr, key, 
		&certblob, &certbloblen, 
		kPGPKeyDBObjProperty_Invalid );
	
    if( IsntPGPError( err ) ) 
		err = sPutPubData( tok, keyidbytes, certblob, certbloblen, 
							keyCrTime_netorder, v );
	
	PGPFreeData( certblob );
	certblob = NULL;

	if( IsPGPError( err ) )
	{
		/* Try again, using only self signatures */
		err = sPgpKeyToMem( context, mgr, key, 
							&certblob, &certbloblen, 
							kPGPSigProperty_KeyID );

		if( IsntPGPError( err ) )  {
			err = sPutPubData( tok, keyidbytes, certblob, certbloblen, 
                               keyCrTime_netorder, v );
		}

		PGPFreeData( certblob );
		certblob = NULL;
	}
	
	return err;
#else
	(void) context;
	(void) key;
	(void) toknumber;
	(void) tok;
	return kPGPError_PublicKeyUnimplemented;
#endif
}

/* Used internally for cleanup only. 
   This function doesn't try to track dependencies between keys */
		PGPError
pgpDeleteKeyOnToken(
	PGPContextRef		context,
	const PGPKeyID		*keyID,
	PGPUInt32			toknumber )
{
#if PGP_WIN32
	PGPToken			*tok;
	PGPByte				*keyidbytes;
	PGPError			err = kPGPError_NoErr;

	/* Don't allow "ANY" token in this call */
	pgpAssert( toknumber != -1 );	

	tok = F.pgpGetNthTokenObject( toknumber );
	if( !tok )
		return kPGPError_SmartCardKeyNotFound;

	keyidbytes = (PGPByte *)pgpGetKeyIDBytes( keyID );
	return pgpTokenObjDeleteKey( tok, keyidbytes, TRUE );
#else
	(void) context;
	(void) keyID;
	(void) toknumber;
	return kPGPError_PublicKeyUnimplemented;
#endif
}



	PGPToken *
pgpTokenFromKeyID(
	const PGPByte *keyid )
{
#if PGP_WIN32
	sLoadTCL();
	if( IsNull( hTCL ) )
		return NULL;
	return F.pgpGetTokenObjectByPrivKeyID( keyid );
#else
	(void)keyid;
	return NULL;
#endif
}

	PGPToken *
pgpTokenGetIndexed(
	PGPUInt32	tokenNum )
{
#if PGP_WIN32
	sLoadTCL();
	if( IsNull( hTCL ) )
		return NULL;
	return F.pgpGetNthTokenObject( tokenNum );
#else
	(void)tokenNum;
	return NULL;
#endif
}

	PGPUInt32
pgpTokenGetCount ( )
{
#if PGP_WIN32
	sLoadTCL();
	if( IsNull( hTCL ) )
		return 0;
	return F.pgpCountTokenObjects( );
#else
	return 0;
#endif
}

	PGPError
pgpGetTokenInfo(
	PGPUInt32		toknumber,
	PGPTokenInfo	*tokenInfo )
{
#if PGP_WIN32
	PGPError		err;
	PGPToken		*tok;

	sLoadTCL();
	if( IsNull( hTCL ) )
		return 0;

	tok = F.pgpGetNthTokenObject( toknumber );
	if( IsNull( tok ) )  {
	    memset( tokenInfo, 0, sizeof(*tokenInfo) );
	    return kPGPError_PublicKeyUnimplemented;
	}

	err = pgpTokenObjGetInfo(tok, tokenInfo);
	return err;
#else
	(void) toknumber;
	memset( tokenInfo, 0, sizeof(*tokenInfo) );
	return 0;
#endif

}

	PGPError
pgpTokenWipe(
	PGPContextRef context, 
	PGPUInt32		toknumber,
	PGPByte const *	passphrase,
	PGPSize			passphraseLength )
{
#if PGP_WIN32
	PGPError		err;
	PGPToken		*tok;
	
	tok = F.pgpGetNthTokenObject( toknumber );
	if( IsNull( tok ) )
		return kPGPError_PublicKeyUnimplemented;
	/* Unlock token */
	err = pgpTokenObjAuth( tok, passphrase, passphraseLength );
	if( IsPGPError( err ) )
		return kPGPError_BadPassphrase;
	err = pgpTokenObjWipe( tok );
	(void)pgpTokenObjDeAuth( tok );
	
	if( IsntPGPError( err ) )
		pgpSyncTokenToKeyDB( context, NULL, TRUE );

	return err;
#else
	(void) context;
	(void) toknumber;
	(void) passphrase;
	(void) passphraseLength;
	return 0;
#endif
}

	PGPError
pgpTokenCheckPassphrase(
	PGPUInt32		toknumber,
	PGPByte const *	passphrase,
	PGPSize			passphraseLength )
{
#if PGP_WIN32
	PGPError		err;
	PGPToken		*tok;

	tok = F.pgpGetNthTokenObject( toknumber );
	if( IsNull( tok ) )
		return kPGPError_PublicKeyUnimplemented;
	/* Unlock token */
	err = pgpTokenObjAuth( tok, passphrase, passphraseLength );
	if( IsPGPError( err ) )
		return kPGPError_BadPassphrase;
	(void)pgpTokenObjDeAuth( tok );
	return kPGPError_NoErr;
#else
	(void) toknumber;
	(void) passphrase;
	(void) passphraseLength;
	return 0;
#endif
}

#if PGP_WIN32
PGPError
sSetKeyTokenNum( PGPKeyDBObj *key, PGPToken *token )  
{
	PGPKeyInfo *kinfo;

	if( !pgpKeyIsOnToken(key)  )
		return kPGPError_SmartCardError;

	pgpAssert(OBJISKEY(key));
	pgpAssert(!pgpFrontEndKey(key));
	pgpAssert(F.pgpGetTokenNum);

	kinfo = pgpKeyToKeyInfo( key );

	kinfo->tokenNum1 = F.pgpGetTokenNum( token ) + 1;

	return kPGPError_NoErr;
}

PGPError
sClearKeyTokenNum( PGPKeyDBObj *key, PGPToken *token )  
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));

	kinfo = pgpKeyToKeyInfo( key );
	kinfo->tokenNum1 = 0;

	return kPGPError_NoErr;
}

#endif

/*
 * Call to iterate over all token objects and update one or all keydbs.
 * If fixedkdb is non-NULL do just that keydb (this is called upon opening
 * the keydb).  If it is null, iterate over all keydbs.
 * If doNotify is true, notify the front end of any changes.
 */
	PGPError
pgpSyncTokenToKeyDB(
	PGPContextRef		context,
	PGPKeyDBRef			fixedkdb,
	PGPBoolean			doNotify )
{
#if PGP_WIN32
	PGPUInt32			tokCount;
	PGPUInt32			i;
	PGPTokenKeyInfo		*keyIDs;
	const KEYIDLEN = sizeof( keyIDs[0].keyid );
	PGPSize				nKeyIDs;
	PGPUInt32			j;
	PGPKeyDBRef			kdb;
	PGPToken			*tok;
	PGPBoolean			didNotify = FALSE;
	static PGPBoolean	reentering = FALSE;

	sLoadTCL();

	/* Can reenter this when we import public blobs */
	if( reentering )
		return kPGPError_NoErr;
	reentering = TRUE;

	tokCount = IsNull( hTCL ) ? 
			0 /* For instance, user selected "No P11 module" */ :
			F.pgpCountTokenObjects( );

	for( i=0; i<tokCount; ++i )
	{
		tok = F.pgpGetNthTokenObject( i );

		/* Copy token public key blobs to key rings */
		keyIDs = pgpTokenObjGetPubKeyIDs ( tok, &nKeyIDs );
		for( j=0; IsntNull(keyIDs) && j<nKeyIDs; j++ )
		{
			PGPKeyID keyid;

	        PGPByte *pk = NULL;
	        PGPUInt32 pklen = 0;
	        PGPBoolean pk_local=TRUE;

	        PGPByte *cert = NULL;
	        PGPUInt32 cert_len = 0;
	        PGPTokenCertInfo *ci = NULL;
	        PGPInt32 ci_n=0;

			PGPNewKeyID( keyIDs[j].keyid, KEYIDLEN, keyIDs[j].alg, &keyid );

			kdb = IsntNull( fixedkdb )
				? fixedkdb
				: pgpContextGetFirstKeyDB( context );
			while( IsntNull( kdb ) )
			{
				if( pgpKeyDBUsesTokens( kdb ) )
				{
					PGPError err;
					PGPKeyDBObjRef key;
	                int k;

					err = pgpGetKeyByKeyID( kdb, &keyid, FALSE, FALSE, &key );
					if( err == kPGPError_ItemNotFound )
					{
			            PGPKeyDBRef newdb_pgp_data=NULL, newdb_certs=NULL;

	                    /* Must copy key to this keydb */

	                    /* Get PGP public key data (one per KeyID) */
	                    if( keyIDs[j].pgpData )  {
	                        if( IsNull( pk ) )
	                        {
	                            pgpTokenDataInfo *di = NULL;

	                            PGPSize data_size;
	                            PGPByte *data = 
	                                 pgpTokenObjGetPublicKeyData ( tok,
	                                    keyIDs[j].keyid, &data_size, &di );

	                            /* Build public key packet form the key stub */
	                            if( !IsNull( data ) ) {
	                                if( !IsNull( di ) )  {
	                                    PGPByte *new_pk;
	                                    
	                                    PGPSize pub_pkt_size;
	                                    PGPByte *pub_pkt = 
	                                        sPgpPubKeyPktFromDataInfo( context, di, &pub_pkt_size );

	                                    pgpTokenFreeTCLMem( di );
	                                    
	                                    if( IsNull(pub_pkt) )  {
	                                        err = kPGPError_OutOfMemory;
	                                        pgpTokenFreeTCLMem( data );
	                                        break;
	                                    }
	                                    
	                                    new_pk = pgpContextMemAlloc( context, pub_pkt_size+data_size, 0 );
	                                    if( IsNull(new_pk) )  {
	                                        err = kPGPError_OutOfMemory;
	                                        PGPFreeData( pub_pkt );
	                                        pgpTokenFreeTCLMem( data );
	                                        break;                                    
	                                    }
	                                    
	                                    pgpCopyMemory( pub_pkt, new_pk, pub_pkt_size );
	                                    pgpCopyMemory( data, new_pk+pub_pkt_size, data_size );
	                                    
	                                    PGPFreeData( pub_pkt );
	                                    pgpTokenFreeTCLMem( data );

	                                    pk = new_pk;
	                                    pklen = pub_pkt_size+data_size;
	                                }
	                                else  {
	                                    pk = data;
	                                    pklen = data_size;
	                                    pk_local = FALSE;
	                                }
	                            }
	                        }
	                        if( pklen!=0 && IsntNull( pk ) )
	                        {
	                            err = pgpImportKeyBinary_internal(
	                                context, pk, pklen, &newdb_pgp_data );
	                        }
	#if PGP_DEBUG
	                        else   {
	                            pgpAssert(0 /* TCL consistency problem */);
	                            pgpAssert( err == kPGPError_ItemNotFound ); 
	                        }
	#endif
	                    }
	                    else
	                        err = kPGPError_NoErr;

	                    /* Now get certificates, if any */
	                    if( IsntPGPError(err) )  {
	                        if( IsNull(ci) )  
	                            ci = pgpTokenObjGetX509Certs(tok, keyIDs+j, &ci_n );
	                        
	                        for( k=0; k<ci_n; k++ )  {
	                            PGPSize certlen=0;
	                            PGPByte *cert = pgpTokenObjGetX509Cert( tok, ci+k, &certlen );
	                            if( certlen!=0 && IsntNull( cert ) )
	                            {
									PGPKeyDBRef db=NULL;

									err = pgpDecodeX509Cert( cert, certlen, 
									    context, NULL, &db );
									pgpTokenFreeTCLMem(cert);

									if( ! IsNull(db) )  {
									    if( IsNull(newdb_certs) )  
									        PGPNewKeyDB( context, &newdb_certs );
									    PGPCopyKeys( PGPPeekKeyDBRootKeySet(db), newdb_certs, NULL);
									    PGPFreeKeyDB( db );
									}

									if( IsPGPError(err) )
									    break;
	                            }
	                        }
	                    }

						if( IsntPGPError( err ) )
						{
	                        if( !IsNull(newdb_pgp_data) )
	                            PGPCopyKeys( PGPPeekKeyDBRootKeySet(newdb_pgp_data), kdb, NULL);
	                        if( !IsNull(newdb_certs) )
						        PGPCopyKeys( PGPPeekKeyDBRootKeySet(newdb_certs), kdb, NULL);
						}

	                    if( !IsNull(newdb_pgp_data) )
	                        PGPFreeKeyDB( newdb_pgp_data );
	                    if( !IsNull(newdb_certs) )
	                        PGPFreeKeyDB( newdb_certs );
					}
				}
				/* Only one iteration if given a keydb */
				if( IsntNull( fixedkdb ) )
					break;

				/* Iterate over all if fixedkdb is null */
				kdb = pgpKeyDBNextKeyDB( kdb );
			}

	        if( ! IsNull(pk) )  {
	            if( pk_local )
	                PGPFreeData( pk );
	            else
	                pgpTokenFreeTCLMem( pk );
	        }
	            
	        pgpTokenFreeTCLMem( ci );
		}
	    pgpTokenFreeTCLMem(keyIDs);

		/* Mark private keys as belonging to token */
		keyIDs = pgpTokenObjGetPrivKeyIDs ( tok, &nKeyIDs );
		for( j=0; IsntNull(keyIDs) && j<nKeyIDs ; j++ )
		{
			PGPKeyID keyid;

			PGPNewKeyID( keyIDs[j].keyid, KEYIDLEN, keyIDs[j].alg , &keyid );

			kdb = IsntNull( fixedkdb )
				? fixedkdb
				: pgpContextGetFirstKeyDB( context );
			while( IsntNull( kdb ) )
			{
				if( pgpKeyDBUsesTokens( kdb ) )
				{
					PGPError err;
					PGPKeyDBObjRef key;

					err = pgpGetKeyByKeyID( kdb, &keyid, FALSE, FALSE, &key );
					if( IsntPGPError( err ) && IsntNull( key ) )
					{
						pgpKeyOnToken( key );
						sSetKeyTokenNum( key, tok );
						if( doNotify )
						{
							pgpSetPendingNotify( NULL, key, NULL, NULL );
							didNotify = TRUE;
						}
					}
				}
				/* Only one iteration if given a keydb */
				if( IsntNull( fixedkdb ) )
					break;
				/* Else iterate over all */
				kdb = pgpKeyDBNextKeyDB( kdb );
			}
		}
	    pgpTokenFreeTCLMem(keyIDs);
	    keyIDs = NULL;
	}

	/* Now check for any keys no longer on tokens */
	kdb = IsntNull( fixedkdb )
		? fixedkdb
		: pgpContextGetFirstKeyDB( context );
	while( IsntNull( kdb ) )
	{
		if( pgpKeyDBUsesTokens( kdb ) )
		{
			PGPKeyDBObjRef key;
			PGPKeyIterRef kiter;

			PGPNewKeyIterFromKeyDB( kdb, &kiter );
			while( IsntPGPError( PGPKeyIterNextKeyDBObj( kiter,
									kPGPKeyDBObjType_Key, &key )))
			{
				PGPByte alg;
				PGPKeyID keyid;
				PGPByte *keyidbytes;

				if( !pgpKeyIsOnToken( key ) )
					continue;
				pgpKeyID8( key, &alg, &keyid );
				if( alg != kPGPPublicKeyAlgorithm_RSA )
					continue;
				keyidbytes = (PGPByte *)pgpGetKeyIDBytes( &keyid );
				if( IsntNull( hTCL ) )  {
					tok = F.pgpGetTokenObjectByPrivKeyID( keyidbytes );
					if( IsntNull( tok ) )
						continue;
				}
				/* Key is no longer on any tokens */
				pgpKeyOffToken( key );
				sClearKeyTokenNum( key, tok );
				if( doNotify )
				{
					pgpSetPendingNotify( NULL, key, NULL, NULL );
					didNotify = TRUE;
				}
			}
			PGPFreeKeyIter( kiter );
		}
		/* Only one iteration if given a keydb */
		if( IsntNull( fixedkdb ) )
			break;

		/* Calculate trust changes due to new token */
		/* Not if called with fixedkdb, caller will do it then */
		PGPCalculateTrust( PGPPeekKeyDBRootKeySet( kdb ), NULL );

		/* Else iterate over all */
		kdb = pgpKeyDBNextKeyDB( kdb );
	}

	/* Force at least one notify if requested */
	if( doNotify && !didNotify )
	{
		kdb = IsntNull( fixedkdb )
			? fixedkdb
			: pgpContextGetFirstKeyDB( context );
		while( IsntNull( kdb ) )
		{
			if( pgpKeyDBUsesTokens( kdb ) )
			{
				PGPKeyDBObjRef key =
					pgpFirstKeyInKeySet( PGPPeekKeyDBRootKeySet( kdb ) );
				pgpSetPendingNotify( NULL, key, NULL, NULL );
				didNotify = TRUE;
			}
			if( IsntNull( fixedkdb ) )
				break;
			kdb = pgpKeyDBNextKeyDB( kdb );
		}
	}

	reentering = FALSE;

#else /* PGP_WIN32 */
	(void) context;
	(void) fixedkdb;
	(void) doNotify;
#endif /* PGP_WIN32 */

	return kPGPError_NoErr;
}

/* This is called to query all tokens and copy any new keys to keydbs */
	PGPError
pgpPollTokens(
	PGPContextRef	context,
	PGPMutex_t		*mutex )
{
#if PGP_WIN32

	sLoadTCL();

	if( IsntNull( context ) )
	{
		PGPBoolean changes;

		/* Allow us to make SDK calls safely */
		PGPMutexLock( mutex );

		changes = FALSE;

		/* Check first if TCL was unloaded, 
		   while we were waiting on the mutex. 
		   ( serialized on the same mutex ) */
		if ( IsntNull( hTCL ) ) { 
			if( F.pgpAcquireAllTokens() != 0 )  {
				changes = TRUE;
				sHadTcl = TRUE;
			}
		}
		else if( sHadTcl )  {
			changes = TRUE;
			sHadTcl = FALSE;
		}

		/* Some change to token list */
		if( changes )  {

			/* Make notify tell all clients */
			pgpContextSetConnectRef( context, kPGPConnectRef_Null );

			pgpSyncTokenToKeyDB( context, NULL, TRUE );
		}

		PGPMutexUnlock( mutex );
	}


#else /* PGP_WIN32 */
	(void) context;
	(void) mutex;
#endif /* PGP_WIN32 */
	return kPGPError_NoErr;
}

	PGPError
pgpUnloadTCL()
{
#if PGP_WIN32
	if( IsntNull(hTCL) )
	{
		HANDLE h = hTCL;

		hTCL = NULL;

		F.pgpReleaseAllTokens();
		memset( &F, 0, sizeof(F) );
		FreeLibrary( h );

	}
	inited = 0;
	sLoadTclTries = 0;
	sCustomTcl = -1;
	sP11Module[0] = '\0';

#endif /* PGP_WIN32 */
	return kPGPError_NoErr;
}

	PGPError 
pgpTokenFreeTCLMem(
	void			*p )
{
#if PGP_WIN32
	pgpAssert( F.pgpFreeMem != NULL );

	if( p )
		F.pgpFreeMem( p );
#else
	(void) p;
#endif
	return kPGPError_NoErr;
}

